# Setup script for LSTM Unroller (Windows PowerShell)
# This script creates a virtual environment and installs all dependencies

# Stop on errors
$ErrorActionPreference = "Stop"

Write-Host "========================================" -ForegroundColor Cyan
Write-Host "LSTM Unroller Setup Script" -ForegroundColor Cyan
Write-Host "========================================" -ForegroundColor Cyan
Write-Host ""

# Get script directory
$ScriptDir = Split-Path -Parent $MyInvocation.MyCommand.Path
Set-Location $ScriptDir

Write-Host "Working directory: $ScriptDir"
Write-Host ""

# Check Python installation
Write-Host "Checking Python installation..."
try {
    $pythonVersion = python --version 2>&1
    if ($pythonVersion -match "Python (\d+\.\d+\.\d+)") {
        $version = $matches[1]
        Write-Host "Found Python $version" -ForegroundColor Green
        
        # Check if version is >= 3.8
        $versionParts = $version.Split('.')
        $major = [int]$versionParts[0]
        $minor = [int]$versionParts[1]
        
        if ($major -lt 3 -or ($major -eq 3 -and $minor -lt 8)) {
            Write-Host "Error: Python 3.8 or higher required (found $version)" -ForegroundColor Red
            exit 1
        }
    }
} catch {
    Write-Host "Error: Python not found" -ForegroundColor Red
    Write-Host "Please install Python 3.8 or higher from https://www.python.org/" -ForegroundColor Yellow
    exit 1
}

Write-Host ""

# Create virtual environment
$VenvDir = "venv"

if (Test-Path $VenvDir) {
    Write-Host "Virtual environment already exists" -ForegroundColor Yellow
    $response = Read-Host "Do you want to recreate it? (y/N)"
    if ($response -match "^[Yy]$") {
        Write-Host "Removing existing virtual environment..."
        Remove-Item -Recurse -Force $VenvDir
    } else {
        Write-Host "Using existing virtual environment"
    }
}

if (-not (Test-Path $VenvDir)) {
    Write-Host "Creating virtual environment..."
    python -m venv $VenvDir
    Write-Host "Virtual environment created" -ForegroundColor Green
} else {
    Write-Host "Virtual environment ready"
}

Write-Host ""

# Activate virtual environment
Write-Host "Activating virtual environment..."
$activateScript = Join-Path $VenvDir "Scripts\Activate.ps1"

# Check if execution policy allows running scripts
$executionPolicy = Get-ExecutionPolicy -Scope CurrentUser
if ($executionPolicy -eq "Restricted") {
    Write-Host "Setting execution policy to allow script execution..." -ForegroundColor Yellow
    Set-ExecutionPolicy -ExecutionPolicy RemoteSigned -Scope CurrentUser -Force
}

& $activateScript

# Upgrade pip
Write-Host "Upgrading pip..."
python -m pip install --upgrade pip --quiet

# Check pip version
$pipVersion = pip --version
if ($pipVersion -match "pip (\d+\.\d+\.\d+)") {
    Write-Host "pip $($matches[1]) installed" -ForegroundColor Green
}

Write-Host ""

# Install requirements
if (Test-Path "requirements.txt") {
    Write-Host "Installing dependencies from requirements.txt..."
    Write-Host "This may take a few minutes..."
    pip install -r requirements.txt
    Write-Host ""
    Write-Host "Dependencies installed successfully!" -ForegroundColor Green
} else {
    Write-Host "Error: requirements.txt not found" -ForegroundColor Red
    exit 1
}

Write-Host ""
Write-Host "========================================" -ForegroundColor Cyan
Write-Host "Setup Complete!" -ForegroundColor Cyan
Write-Host "========================================" -ForegroundColor Cyan
Write-Host ""
Write-Host "To activate the virtual environment, run:" -ForegroundColor Yellow
Write-Host "  .\venv\Scripts\Activate.ps1" -ForegroundColor Green
Write-Host ""
Write-Host "To test the installation, run:" -ForegroundColor Yellow
Write-Host "  python unroll_lstm_layers.py --help" -ForegroundColor Green
Write-Host ""
Write-Host "To run all tests (requires bash), install Git Bash and run:" -ForegroundColor Yellow
Write-Host "  cd tests && ./run_all_tests.sh" -ForegroundColor Green
Write-Host ""
Write-Host "Alternatively, run individual test suites:" -ForegroundColor Yellow
Write-Host "  python tests/test_with_keras2.py" -ForegroundColor Green
Write-Host "  python tests/test_with_keras3.py" -ForegroundColor Green
Write-Host ""
Write-Host "To deactivate the virtual environment when done:" -ForegroundColor Yellow
Write-Host "  deactivate" -ForegroundColor Green
Write-Host ""

# Keep PowerShell window open
Write-Host "Press any key to continue..."
$null = $Host.UI.RawUI.ReadKey("NoEcho,IncludeKeyDown")
