# LSTM Unroller for TFLite Micro

## Quick Start

### 1. Setup Environment

**macOS/Linux:**
```bash
cd lstm_unroller
source ./setup.sh
```

**Windows PowerShell:**
```powershell
cd lstm_unroller
.\setup.ps1
```

The setup script will:
- Create and activate a Python virtual environment
- Upgrade pip
- Install all dependencies (TensorFlow 2.19, tf-keras, h5py, numpy)

### 2. Basic Usage

```bash
# Unroll LSTM layers with Keras 2 (default)
python unroll_lstm_layers.py input.h5 output.h5

# Unroll LSTM layers with Keras 3
python unroll_lstm_layers.py input.h5 output.h5 --keras-version 3

# Preview changes without saving (dry run)
python unroll_lstm_layers.py input.h5 --dry-run

# Force unrolling regardless of timesteps
python unroll_lstm_layers.py input.h5 output.h5 --force

# Custom timestep limit (default: 20)
python unroll_lstm_layers.py input.h5 output.h5 --max-timesteps 15
```

### 3. Verify Installation

```bash
# Show help
python unroll_lstm_layers.py --help
```

---

## Dependencies
All dependencies are automatically installed by the setup script:
- `tensorflow==2.19.0`
- `tf-keras==2.19.0` (for Keras 2 compatibility)
- `h5py==3.14.0`
- `numpy==1.26.4`

---

## Usage Guide

### Command Line Options

```
python unroll_lstm_layers.py INPUT OUTPUT [OPTIONS]

Required arguments:
  INPUT              Input H5 model file
  OUTPUT             Output H5 model file (default: INPUT-unrolled.h5)

Optional arguments:
  --keras-version {2,3}     Keras version (default: 2)
                            Use 3 for Keras 3 models
  --max-timesteps N         Only unroll if timesteps <= N (default: 20)
  --force                   Unroll regardless of timestep count
  --dry-run                 Show what would be changed without saving
  --help                    Show help message
```

### Examples

**Example 1: Basic unrolling (Keras 2)**
```bash
python unroll_lstm_layers.py my_model.h5 my_model_unrolled.h5
```

**Example 2: Unroll Keras 3 model**
```bash
python unroll_lstm_layers.py model_keras3.h5 output.h5 --keras-version 3
```

**Example 3: Preview changes first**
```bash
# Dry run to see what will be changed
python unroll_lstm_layers.py model.h5 --dry-run

# If satisfied, run the actual conversion
python unroll_lstm_layers.py model.h5 model_unrolled.h5
```

**Example 4: Unroll only very short sequences**
```bash
python unroll_lstm_layers.py model.h5 output.h5 --max-timesteps 10
```

**Example 5: Force unroll long sequences**
```bash
python unroll_lstm_layers.py model.h5 output.h5 --force
```

---

## What Gets Unrolled?

### ✅ Supported (Will Unroll)
- **LSTM layers** with ≤20 timesteps (default limit)
- **Bidirectional LSTM layers** with ≤20 timesteps

### ⏭️ Skipped (Won't Unroll)
- **GRU layers**
- **SimpleRNN layers**
- **LSTM layers** already set to `unroll=True`
- **LSTM layers** with >20 timesteps (unless `--force` is used)
- **LSTM layers** with variable timesteps (None)

---

## Understanding the Output

When you run the script, you'll see:

```
================================================================================
LSTM Layer Unroller for TFLite Micro
================================================================================

Step 1: Loading model
--------------------------------------------------------------------------------
Model loaded: my_model
  Input shape: (None, 10, 32)
  Output shape: (None, 2)
  Total layers: 8

Step 2: Analyzing RNN layers
--------------------------------------------------------------------------------
Found 2 RNN layer(s):

  [UNROLL] Layer 3: lstm_1
    Type: LSTM
    Current unroll: False
    Timesteps: 10
    Decision: 10 timesteps <= 20 limit

  [UNROLL] Layer 4: lstm_2
    Type: LSTM
    Current unroll: False
    Timesteps: 10
    Decision: 10 timesteps <= 20 limit

Will unroll 2 layer(s)

Step 3: Extracting custom attributes
--------------------------------------------------------------------------------
Extracted 4 custom attributes

Step 4: Recreating model with unrolled layers
--------------------------------------------------------------------------------
New model created

Step 5: Copying weights
--------------------------------------------------------------------------------
Copied weights for 8 layers

Step 6: Saving model
--------------------------------------------------------------------------------
Model saved to: my_model_unrolled.h5

Step 7: Restoring custom attributes
--------------------------------------------------------------------------------
Restored 4 custom attributes

Step 8: Fixing model_config JSON
--------------------------------------------------------------------------------
Updated 2 layer(s) in model_config

Step 9: Validation
--------------------------------------------------------------------------------
Output model loads successfully

Verifying unroll status:
  [OK] lstm_1: unroll=True
  [OK] lstm_2: unroll=True

Validating inference:
  Max difference: 0.0000000000
  Mean difference: 0.0000000000
  Outputs match perfectly!

================================================================================
SUCCESS
================================================================================

Unrolled 2 RNN layer(s)
Preserved 4 custom attributes
Output: my_model_unrolled.h5
```

---

## Keras 2 vs Keras 3

### Keras 2 (Default)
- Uses `tf-keras` package
- Standard H5 loading
- Set with `--keras-version 2` (default)

### Keras 3
- Uses TensorFlow's built-in Keras 3
- Automatically handles `time_major` attribute incompatibility
- Set with `--keras-version 3`

**Note:** The script uses Keras 2 by default. If your model was built with Keras 3 set `--keras-version 3`.

---

## Troubleshooting

### Issue: "Python not found"
**Solution:** Install Python 3.8+ from [python.org](https://www.python.org/)

### Issue: "No RNN layers found"
**Solution:** Your model may not contain LSTM layers, or they may be in a custom wrapper. Check your model architecture.

---

## Limitations

1. **Only LSTM and Bidirectional LSTM** are supported (TFLite Micro limitation)
2. **Fixed-length sequences only** - Variable length sequences (None timesteps) cannot be unrolled
3. **Keras Sequential and Functional models** - Subclassed models are not supported

---

## FAQ

**Q: Will this change my model's accuracy?**  
A: No. Unrolling is a graph optimization that doesn't change the computation. The script validates that outputs are numerically identical.

**Q: Can I unroll models with very long sequences (>50 timesteps)?**  
A: You can with `--force`, but it's not recommended. The resulting model will be very large and may not fit on embedded devices.